<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Package;
use App\Models\Review;
use Illuminate\Http\Request;

class ReviewController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $reviews = user()->reviews()->with('reviewable')->latest()->get();

        return view('user.review.index', compact('reviews'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        abort(404);
        $review = new Review;

        return view('user.review.create', compact('review'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // return
        $data = $request->validate([
            'rating' => 'required|numeric|min:1|max:5',
            'content' => 'required|string',
            'package_id' => 'required|exists:packages,id',
        ]);

        $package = Package::findOrFail($data['package_id']);
        $user = user();
        try {

            if (! $user->bookings()->where('package_id', $package->id)->exists()) {
                return back()->with('error', 'You have not booked this package.');
            }

            if ($package->reviews()->where('user_id', $user->id)->exists()) {
                return back()->with('error', 'You have already reviewed this package.');
            }

            $package->reviews()->create([
                'user_id' => $user->id,
                'content' => $data['content'],
                'rating' => $data['rating'],
            ]);

            return back()->with('success', 'Review sent successfully.');
        } catch (\Exception $e) {
            return back()->with('error', 'Review could not be created.');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Review $review)
    {
        abort(404);

        return view('user.review.show', compact('review'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Review $review)
    {
        return view('user.review.create', compact('review'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Review $review)
    {
        $data = $request->validate([
            'rating' => 'required|numeric|min:0|max:4',
            'content' => 'required|string',
        ]);
        $data['rating'] = $request->rating + 1;

        if (user()->hasRole('agent')) {
            $data['published'] = $request->has('published');
        }

        try {
            $review->update($data);

            return redirect()->route('user.review.index')->with('success', 'Review updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Review could not be updated.');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Review $review)
    {
        try {
            $review->delete();

            return response()->json(['message' => 'Review deleted successfully.', 'status' => 'success']);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Review could not be deleted.', 'status' => 'error'], 500);
        }
    }
}
